package zutils

import (
	"reflect"
	"sort"

	"golang.org/x/exp/constraints"
)

func ArrayUnique[T comparable](s []T) []T {
	keys := make(map[T]bool)
	var list []T

	for _, entry := range s {
		if _, value := keys[entry]; !value {
			keys[entry] = true
			list = append(list, entry)
		}
	}

	return list
}

func InArray[T comparable](needle T, haystack []T) bool {
	for _, v := range haystack {
		if needle == v {
			return true
		}
	}

	return false
}

func ArrayRemove[T comparable](slice []T, s int) []T {
	return append(slice[:s], slice[s+1:]...)
}

func ArrayColumn[T comparable](input []map[string]T, columnKey string) []T {
	columns := make([]T, 0, len(input))

	for _, val := range input {
		if v, ok := val[columnKey]; ok {
			columns = append(columns, v)
		}
	}

	return columns
}

func ArrayCombine[K, V comparable](keys []K, values []V) map[K]V {
	if len(keys) != len(values) {
		return nil
	}

	m := make(map[K]V, len(keys))
	for i, v := range keys {
		m[v] = values[i]
	}

	return m
}

func ArrayDiff[T comparable](array1, array2 []T) []T {
	var res []T
	for _, v := range array1 {
		if !InArray(v, array2) {
			res = append(res, v)
		}
	}

	return res
}

func ArrayIntersect[T comparable](array1, array2 []T) []T {
	var res []T
	for _, v := range array1 {
		if InArray(v, array2) {
			res = append(res, v)
		}
	}

	return res
}

func ArrayFlip(input any) any {
	if input == nil {
		return nil
	}
	val := reflect.ValueOf(input)
	if val.Len() == 0 {
		return nil
	}
	res := make(map[any]any, val.Len())
	switch val.Kind() {
	case reflect.Slice, reflect.Array:
		for i := 0; i < val.Len(); i++ {
			res[val.Index(i).Interface()] = i
		}

		return res
	case reflect.Map:
		for _, k := range val.MapKeys() {
			res[val.MapIndex(k).Interface()] = k.Interface()
		}

		return res
	}

	return nil
}

func ArrayKeys(input any) any {
	if input == nil {
		return nil
	}
	val := reflect.ValueOf(input)
	if val.Len() == 0 {
		return nil
	}

	switch val.Kind() {
	case reflect.Slice, reflect.Array:
		var res []int
		for i := 0; i < val.Len(); i++ {
			res = append(res, i)
		}

		return res
	case reflect.Map:
		var res []string
		for _, k := range val.MapKeys() {
			res = append(res, k.String())
		}
		sort.SliceStable(res, func(i, j int) bool {
			return res[i] < res[j]
		})

		return res
	}

	return nil
}

func ArrayKeyExists[K, V comparable](k K, m map[K]V) bool {
	return KeyExists(k, m)
}

func KeyExists[K, V comparable](k K, m map[K]V) bool {
	_, ok := m[k]

	return ok
}

func ArrayFilter(input any, callback func(any) bool) any {
	if input == nil {
		return nil
	}
	val := reflect.ValueOf(input)
	if val.Len() == 0 {
		return nil
	}
	if callback == nil {
		callback = func(v any) bool {
			return v != nil
		}
	}
	switch val.Kind() {
	case reflect.Slice, reflect.Array:
		var res []any
		for i := 0; i < val.Len(); i++ {
			v := val.Index(i).Interface()
			if callback(v) {
				res = append(res, v)
			}
		}

		return res
	case reflect.Map:
		res := make(map[any]any)
		for _, k := range val.MapKeys() {
			v := val.MapIndex(k).Interface()
			if callback(v) {
				res[k.Interface()] = v
			}
		}

		return res
	}

	return input
}

func ArrayPad[T comparable](array []T, size int, value T) []T {
	if size == 0 || (size > 0 && size < len(array)) || (size < 0 && size > -len(array)) {
		return array
	}
	n := size
	if size < 0 {
		n = -size
	}
	n -= len(array)
	tmp := make([]T, n)
	for i := 0; i < n; i++ {
		tmp[i] = value
	}
	if size > 0 {
		return append(array, tmp...)
	}

	return append(tmp, array...)
}

func ArrayPop[T comparable](s *[]T) T {
	var t T
	if s == nil || len(*s) == 0 {
		return t
	}

	ep := len(*s) - 1
	e := (*s)[ep]
	*s = (*s)[:ep]

	return e
}

func ArrayPush[T comparable](s *[]T, elements ...T) int {
	if s == nil {
		return 0
	}
	*s = append(*s, elements...)

	return len(*s)
}

func ArrayShift[T comparable](s *[]T) T {
	var t T
	if s == nil || len(*s) == 0 {
		return t
	}

	f := (*s)[0]
	*s = (*s)[1:]

	return f
}

func ArrayUnshift[T comparable](s *[]T, elements ...T) int {
	if s == nil {
		return 0
	}
	*s = append(elements, *s...)

	return len(*s)
}

func ArrayReverse[T comparable](s []T) []T {
	for i, j := 0, len(s)-1; i < j; i, j = i+1, j-1 {
		s[i], s[j] = s[j], s[i]
	}

	return s
}

func ArraySlice[T comparable](array []T, offset, length uint) []T {
	if offset > uint(len(array)) {
		return nil
	}
	end := offset + length
	if end < uint(len(array)) {
		return array[offset:end]
	}

	return array[offset:]
}

func ArraySum[T constraints.Ordered](array []T) T {
	var sum T
	for _, v := range array {
		sum += v
	}

	return sum
}

func Sort[T constraints.Ordered](array []T) []T {
	if len(array) == 0 {
		return array
	}

	sort.Slice(array, func(i int, j int) bool {
		return array[i] < array[j]
	})

	return array
}

func Rsort[T constraints.Ordered](array []T) []T {
	if len(array) == 0 {
		return array
	}

	sort.Slice(array, func(i int, j int) bool {
		return array[i] > array[j]
	})

	return array
}
