package zdb

import (
	"time"
)

type Stat struct {
	ID   string
	Mode connMode
	Host string
	Port uint16

	AcquireCount            int64
	AcquireDuration         time.Duration
	AcquiredConns           int32
	CanceledAcquireCount    int64
	ConstructingConns       int32
	EmptyAcquireCount       int64
	IdleConns               int32
	MaxConns                int32
	TotalConns              int32
	NewConnsCount           int64
	MaxLifetimeDestroyCount int64
	MaxIdleDestroyCount     int64
}

func (d *Pool) StatMaster() *Stat {
	return d.Stat(d.SrvMaster)
}

func (d *Pool) StatPool() []*Stat {
	out := make([]*Stat, 0)
	for _, q := range append(d.SrvSync, d.SrvAsync...) {
		out = append(out, d.Stat(q))
	}

	if d.SrvMaster != nil {
		return append(out, d.Stat(d.SrvMaster))
	}

	return out
}

func (d *Pool) StatPoolTotal() *Stat {
	m := &Stat{ID: "pool", Mode: connModePool}

	for _, s := range d.StatPool() {
		m.AcquireCount += s.AcquireCount
		m.AcquireDuration += s.AcquireDuration
		m.AcquiredConns += s.AcquiredConns
		m.CanceledAcquireCount += s.CanceledAcquireCount
		m.ConstructingConns += s.ConstructingConns
		m.EmptyAcquireCount += s.EmptyAcquireCount
		m.IdleConns += s.IdleConns
		m.MaxConns += s.MaxConns
		m.TotalConns += s.TotalConns
		m.NewConnsCount += s.NewConnsCount
		m.MaxLifetimeDestroyCount += s.MaxLifetimeDestroyCount
		m.MaxIdleDestroyCount += s.MaxIdleDestroyCount
	}

	return m
}

func (d *Pool) Stat(q *Conn) *Stat {
	s := q.Stat()

	return &Stat{
		ID:                      q.ToString(),
		Mode:                    q.Mode,
		Host:                    q.Config().ConnConfig.Host,
		Port:                    q.Config().ConnConfig.Port,
		AcquireCount:            s.AcquireCount(),
		AcquireDuration:         s.AcquireDuration(),
		AcquiredConns:           s.AcquiredConns(),
		CanceledAcquireCount:    s.CanceledAcquireCount(),
		ConstructingConns:       s.ConstructingConns(),
		EmptyAcquireCount:       s.EmptyAcquireCount(),
		IdleConns:               s.IdleConns(),
		MaxConns:                s.MaxConns(),
		TotalConns:              s.TotalConns(),
		NewConnsCount:           s.NewConnsCount(),
		MaxLifetimeDestroyCount: s.MaxLifetimeDestroyCount(),
		MaxIdleDestroyCount:     s.MaxIdleDestroyCount(),
	}
}
